/** @file   level.cpp
 * @brief   Implementation of Level - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#include "level.h"
#include "www_assert.h"
#include <fstream>
#include "utils.h"
#include "filehasher.h"
#include "door.h"
#include "www_map.h"
using std::ifstream;
using std::istream;
using std::string;

namespace WeWantWar {


/** Constructor
 */
Level::Level() :
  weather( 0 ),
  timebonus( 0 ),
  nightModeOn(false)
{
  playerStartingPos.plaX = 0;
  playerStartingPos.plaY = 0;
  playerStartingPos.plaAngle = 0;
  playerStartingPos.plaID = 0;
}



/** Destructor
 */
Level::~Level()
{
  this->cleanup();
}


/** Loads the level
 */
int Level::load( const std::string& levelfile )
{
  // First we check the hash-code.
  eng2d::FileHasher hf( levelfile );
  if ( hf.check() == true ) {
    return EHashCheckFailed;
  }
  
  // Open the description file
  ifstream fin( levelfile.c_str() );
  if ( !fin ) {
    return EFailedToOpen;
  }
  
  if ( Utils::searchForString( fin, "<wewantwar_level>" ) == true ) {
    fin.close();
    return EInvalidFileFormat;
  }
  
  // Parse level-file.
  if ( this->parseLevelFile( fin ) != 0 ) {
    fin.close();
    return EParseError;
  }
  
  fin.close();
  
  if ( Map::load( mapfile ) != 0 ) {
    return EFmpMapFailure;
  }
  return ENone;
}


/** Returns an objective
 */
Objective* Level::getObjective( int index ) const
{
  WWW_ASSERT( index >= 0 );
  WWW_ASSERT( index < m_objectives.size() );
  
  return m_objectives.at( index );
}



/** Returns the number of objectives
 */
int Level::objectiveCount() const
{
  return m_objectives.size();
}



/** Returns the number of unfinished objectives
 */
int Level::unfinishedObjectives() const
{
  int objCount = 0;
  for ( int i=0; i < m_objectives.size(); i++ ) {
    if ( m_objectives.at(i)->state() != Objective::ACCOMPLISHED ) {
      objCount += 1;
    }
  }
  return objCount;
}



/** Returns the route
 */
Route Level::getRoute( const std::string& routename ) const
{
  for ( int i=0; i< routeTable.size(); i++ ) {
    if ( routeTable.at(i).name() == routename ) {
      return routeTable.at(i);
    }
  }
  return Route( "NO_ROUTE" );
}




///
/// Private or Protected methods
/// ============================

/** Cleans up
 */
void Level::cleanup()
{
  routeTable.clear();
  storylines.clear();
  
  for ( int i=0; i < m_objectives.size(); i++ ) {
    if ( m_objectives.at(i) != 0 ) {
      delete m_objectives.at(i);
      m_objectives.at(i) = 0;
    }
  }
  m_objectives.clear();
}


/** Reads the levelfile
 */
int Level::parseLevelFile( istream& rIn )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }

    string tmp;
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );

    } else if ( tmp == "<general_information>" ) {
      if ( this->readGeneralInformation( rIn ) != 0 ) {
        return -1;
      }

    } else if ( tmp == "<player>" ) {
      if ( this->readPlayerInformation( rIn ) != 0 ) {
        return -1;
      }

    } else if ( tmp == "<doors>" ) {
      if ( Door::initDoorBlocks( rIn ) != 0 ) {
        return -1;
      }

    } else if ( tmp == "<objective>" ) {
      // Load the objective.
      Objective* pO = new Objective();
      if ( pO->readData( rIn ) != 0 ) {
        return -1;
      }
      m_objectives.push_back( pO );

    } else if ( tmp == "<story>" ) {
      if ( this->readStory( rIn ) != 0 ) {
        return -1;
      }

    } else if ( tmp == "<include>" ) {
      if ( this->includeFile( rIn ) != 0 ) {
        return -1;
      }
      
    } else if ( tmp == "</wewantwar_level>" || tmp == "[END]" ) {
      break;

    } else if ( tmp == "<wewantwar_route>" ) {
      Route fooRoute( "NO_ROUTE" );
      if ( fooRoute.readRoute( rIn ) != 0 ) {
        return -1;
      }
      this->routeTable.push_back( fooRoute );
      
    } else {
      int ret = alert( "Map Description File", "Unkown element", tmp.c_str(),
                       "Continue", "Quit", 0,0 );
      if ( ret == 2 ) {
        return -1;
      }
      
    }

  }
  return 0;
}



/** Reads the general information.
 */
int Level::readGeneralInformation( istream& rIn )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "</general_information>" ) {
      break;
      
    } else if ( tmp == "mapfile:" ) {
      rIn >> mapfile;
      
    } else if ( tmp == "weather:" ) {
      rIn >> weather;
      
    } else if ( tmp == "music:" ) {
      std::string tmpMusicFile;
      rIn >> tmpMusicFile;
      musicfile.push_back( tmpMusicFile );
      
    } else if ( tmp == "timebonus:" ) {
      rIn >> timebonus;
      
    } else if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );

    } else if ( tmp == "nightmode:" ) {
      rIn >> tmp;
      if ( tmp == "on" ) {
        nightModeOn = true;
      } else {
        nightModeOn = false;
      }
      
    }
  }
  
  // We load the mapfile
  if ( Map::load( mapfile ) != 0 ) {
    return -1;
  }
  return 0;
}



/** Reads the player's information
 */
int Level::readPlayerInformation( istream& rIn )
{
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "pos:" ) {
      rIn >> playerStartingPos.plaX;
      rIn >> playerStartingPos.plaY;
      
    } else if ( tmp == "angle:" ) {
      rIn >> playerStartingPos.plaAngle;
      
    } else if ( tmp == "id:" ) {
      rIn >> playerStartingPos.plaID;
      
    } else if ( tmp == "</player>" ) {
      return 0;
      
    }
  }
}



/** Reads the level story.
 */
int Level::readStory( istream& rIn )
{
  char storybuffer[256];
  
  while ( true ) {
    if ( rIn.eof() == true ) {
      return -1;
    }
    
    string tmp;
    rIn >> tmp;
    if ( tmp == "#" ) {
      rIn.ignore( 4096, '\n' );
      
    } else if ( tmp == "</story>" ) {
      return 0;
      
    } else if ( tmp == "line:" ) {
      rIn.getline( storybuffer, 255 );
      string st( storybuffer );
      if ( st == " <br>" ) {
        st = "";
      }
      storylines.push_back( st );
      
    } else if ( tmp == "background:" ) {
      rIn >> titlepicfile;
      
    } else if ( tmp == "story:" ) {
      rIn >> storypicfile;
      
    }
  }
  return 0;
}



/** Includes another mapfile to this level.
 */
int Level::includeFile( istream& rIn )
{
  string tmp;
  string inctype;
  string incfile;
  rIn >> tmp >> inctype;
  rIn >> tmp >> incfile;
  rIn >> tmp;   // Reads the closing </include> - tag
  
  if ( inctype == "LEVEL_DESCRIPTION" ) {
    eng2d::FileHasher hf( incfile );
    if ( hf.check() == true ) {
      // invalid hashcode
      return -1;
    }
    
    ifstream fin( incfile.c_str() );
    if ( !fin ) {
      alert( "Included file is missing", incfile.c_str(), 0, "ok",0, 0,0 );
      return -1;
    }
    
    if ( Utils::searchForString( fin, "<wewantwar_level>" ) == true ) {
      alert( "Invalid included file", incfile.c_str(), 0, "ok",0, 0,0 );
      fin.close();
      return -1;
    }
    
    int ret = this->parseLevelFile( fin );
    fin.close();
    return ret;
    
  } else if ( inctype == "ROUTEFILE" ) {
    ifstream fin( incfile.c_str() );
    if ( !fin ) {
      return -1;
    }
    if ( Utils::searchForString( fin, "<wewantwar_map_routefile>" ) == true ) {
      fin.close();
      return -1;
    }
    while ( true ) {
      if ( fin.eof() == true ) {
        fin.close();
        return -1;
      }

      fin >> tmp;
      if ( tmp == "#" ) {
        fin.ignore( 4096, '\n' );

      } else if ( tmp == "<wewantwar_route>" ) {
        Route fooRoute( "FOOROUTE" );
        if ( fooRoute.readRoute( fin ) != 0 ) {
          fin.close();
          return -1;
        }
        routeTable.push_back( fooRoute );

      } else if ( tmp == "</wewantwar_map_routefile" ) {
        fin.close();
        return 0;

      } else if ( tmp == "[END]" ) {
        fin.close();
        return 0;
      }
    }
  }
  return 0;
}

} // end of namespace

